package setting

import (
	"fmt"
	"strconv"
	"time"

	"github.com/fsnotify/fsnotify"
	"github.com/spf13/viper"
)

var Conf = new(Config)

type Config struct {
	ServerSetting   *Server                 `mapstructure:"server"`
	AppSetting      *App                    `mapstructure:"app"`
	DatabaseSetting *Database               `mapstructure:"database"`
	LogSetting      *Log                    `mapstructure:"log"`
	PayUrl          *PayUrlDetail           `mapstructure:"payurl"`
	Lakala          *Lakala                 `mapstructure:"lakala"`
}

// Server 服务
type Server struct {
	RunMode      string        `mapstructure:"runmode"`
	HttpPort     int           `mapstructure:"httpport"`
	ReadTimeout  time.Duration `mapstructure:"readtimeout"`
	WriteTimeout time.Duration `mapstructure:"writetimeout"`
}

type App struct {
	Name       string `mapstructure:"name"`
	Version    string `mapstructure:"version"`
	TimeFormat string `mapstructure:"timeformat""`
}

// Database DB
type Database struct {
	Type      string `mapstructure:"type""`
	SystemDB  string `mapstructure:"systemdb"`
}

type Log struct {
	LogSavePath string `mapstructure:"logsavepath"`
	LogSaveName string `mapstructure:"logsavename"`
	LogFileExt  string `mapstructure:"logfileext"`
	Level       string `mapstructure:"level"`
	MaxSize     int    `mapstructure:"max_size"`
	MaxAge      int    `mapstructure:"max_age"`
	MaxBackups  int    `mapstructure:"max_backups"`
}

// PayUrlDetail 网关
type PayUrlDetail struct {
	DomainName string `mapstructure:"domainname"`
}

// 拉卡拉支付
type Lakala struct {
	// 谛宝多多自营 - start
	DbcVersion string `mapstructure:"dbc_version"`
	DbcAppid string `mapstructure:"dbc_appid"`
	DbcSerialNo string `mapstructure:"dbc_serial_no"`
	// 1.聚合收银台(微信H5、支付宝H5)
	DbcMerchantNo1 string `mapstructure:"dbc_merchant_no1"`
	// 2.聚合收银台(微信扫码、支付宝扫码)
	DbcMerchantNo2 string `mapstructure:"dbc_merchant_no2"`
	// 3.聚合主扫(微信JSAPI、微信小程序)
	DbcMerchantNo3 string `mapstructure:"dbc_merchant_no3"`
	DbcTermNo3 string `mapstructure:"dbc_term_no3"`
	// 4.扫码枪
	DbcMerchantNo4 string `mapstructure:"dbc_merchant_no4"`
	DbcTermNo4 string `mapstructure:"dbc_term_no4"`
	// 配置项
	DbcPathPrivateKey string `mapstructure:"dbc_path_private_key"` //私钥
	DbcPathCert string `mapstructure:"dbc_path_cert"` //证书
	// 谛宝多多自营 - end

	// 必康自营 - start
	BkVersion string `mapstructure:"bk_version"`
	BkAppid string `mapstructure:"bk_appid"`
	BkSerialNo string `mapstructure:"bk_serial_no"`
	// 1.聚合收银台(微信H5、支付宝H5)
	BkMerchantNo1 string `mapstructure:"bk_merchant_no1"`
	// 2.聚合收银台(微信扫码、支付宝扫码)
	BkMerchantNo2 string `mapstructure:"bk_merchant_no2"`
	// 3.聚合主扫(微信JSAPI、微信小程序)
	BkMerchantNo3 string `mapstructure:"bk_merchant_no3"`
	BkTermNo3 string `mapstructure:"bk_term_no3"`
	// 4.扫码枪
	BkMerchantNo4 string `mapstructure:"bk_merchant_no4"`
	BkTermNo4 string `mapstructure:"bk_term_no4"`
	// 配置项
	BkPathPrivateKey string `mapstructure:"bk_path_private_key"` //私钥
	BkPathCert string `mapstructure:"bk_path_cert"` //证书
	// 必康自营 - end

	// 必康医生 - start
	SaasVersion string `mapstructure:"saas_version"`
	SaasAppid string `mapstructure:"saas_appid"`
	SaasSerialNo string `mapstructure:"saas_serial_no"`
	// 1.聚合收银台(微信H5、支付宝H5)
	SaasMerchantNo1 string `mapstructure:"saas_merchant_no1"`
	// 2.聚合收银台(微信扫码、支付宝扫码)
	SaasMerchantNo2 string `mapstructure:"saas_merchant_no2"`
	// 3.聚合主扫(微信JSAPI、微信小程序)
	SaasMerchantNo3 string `mapstructure:"saas_merchant_no3"`
	SaasTermNo3 string `mapstructure:"saas_term_no3"`
	// 4.扫码枪
	SaasMerchantNo4 string `mapstructure:"saas_merchant_no4"`
	SaasTermNo4 string `mapstructure:"saas_term_no4"`
	// 配置项
	SaasPathPrivateKey string `mapstructure:"saas_path_private_key"` //私钥
	SaasPathCert string `mapstructure:"saas_path_cert"` //证书
	// 必康医生 - end

	// 拉卡拉接口地址
	UrlCreate string `mapstructure:"url_create"` //聚合收银台(微信H5、支付宝H5、微信扫码、支付宝扫码)
	UrlPreorder string `mapstructure:"url_preorder"` //聚合主扫(微信JSAPI、微信小程序)
	UrlMicropay string `mapstructure:"url_micropay"` //聚合被扫(扫码枪)
	UrlRefund string `mapstructure:"url_refund"` //聚合扫码-退款交易
	UrlTradeRefund string `mapstructure:"url_trade_refund"` //统一退货接口
	UrlOrderState string `mapstructure:"url_order_state"` //聚合扫码-交易查询
}

// Init 支持热修改的viper设置
func Init() error {
	viper.SetConfigFile("conf/config.yaml") // 指定配置文件路径
	err := viper.ReadInConfig()             // 读取配置信息
	if err != nil {                         // 读取配置信息失败
		fmt.Printf("viper.ReadInConfig failed, new_error:%v\n", err)
		return err
	}
	// 将读取的配置信息保存至全局变量Conf
	if err = viper.Unmarshal(Conf); err != nil {
		fmt.Printf("viper.Unmarshal failed, new_error:%v\n", err)
	}
	// 监控配置文件变化
	viper.WatchConfig()
	// 注意！！！配置文件发生变化后要同步到全局变量Conf
	viper.OnConfigChange(func(in fsnotify.Event) {
		fmt.Println("配置文件被修改啦...")
		if err = viper.Unmarshal(Conf); err != nil {
			fmt.Printf("viper.Unmarshal failed, new_error:%v\n", err)
		}
		//fmt.Println(Conf)
	})

	return err
}

func GetPort() string {
	s := strconv.Itoa(Conf.ServerSetting.HttpPort)
	return ":" + s
}
