package pay

import (
	"fmt"
	"github.com/gin-gonic/gin"
	"go.uber.org/zap"
	"strconv"
	"system_pay/controller/base"
	"system_pay/models"
	"system_pay/repository/pay"
)

// 卡拉卡统一支付
type PayController struct {
}

// UnifiedOrder 拉卡拉统一支付
// @Summary 拉卡拉统一支付
// @Description 拉卡拉统一支付
// @Tags 拉卡拉统一支付
// @Accept application/json
// @Produce application/json
// @Param   body    body  models.PlaceAnOrderParamInput   true	 "参数"
// @Param language header string ture "语言类型 zh-CN简体中文 en-US英文 ja-JP日文 默认中文"
// @Success 200
// @router /api/v1/pay/unified_order [post]
func (l *PayController) UnifiedOrder(c *gin.Context) {

	ph := new(models.PlaceAnOrderParamInput)
	//fmt.Println("ContentType="+c.ContentType())
	err := c.ShouldBindJSON(ph)
	if err != nil {
		zap.L().Error(err.Error()) //logs错误日志
		base.ResponseErrorWithMsg(c, base.ServerError)
		return
	}

	// clientIp ip
	ip := c.ClientIP()
	//fmt.Println("ip="+ip)

	// 拉卡拉统一支付
	rtn, err := pay.UnifiedOrder(ph, ip)
	if err != nil {
		zap.L().Error(err.Error())
		//base.ResponseErrorWithMsg(c, base.ServerError)
		base.ResponseErrorMsg(c, err.Error())
		return
	}

	base.ResponseSuccess(c, rtn)
}

// Refund 拉卡拉退款
// @Summary 拉卡拉退款
// @Description 拉卡拉退款
// @Tags 拉卡拉退款
// @Accept application/json
// @Produce application/json
// @Param   orderId   path   string   true	  "退款号:原对账单流水号"
// @Param   operator   path   string   true	  "操作人/退款原因"
// @Param   money   path   string   true	  "退款金额,个位为分"
// @Param language header string ture "语言类型 zh-CN简体中文 en-US英文 ja-JP日文 默认中文"
// @Success 200
// @router /api/v1/pay/refund/{orderId}/{operator}/{money} [put]
func (l *PayController) Refund(c *gin.Context) {

	//ph := new(models.RefundParamInput)
	//err := c.ShouldBindJSON(ph)
	//if err != nil {
	//	zap.L().Error(err.Error())
	//	base.ResponseErrorWithMsg(c, base.ServerError)
	//	return
	//}

	//退款金额,个位为分
	money, err := strconv.ParseFloat(c.Param("money"), 64)
	if err != nil {
		fmt.Println("退款金额转换失败：", err)
		return
	}

	var ph models.RefundParamInput
	ph.RefundNo = c.Param("orderId") //退款号:原对账单流水号
	ph.RefundAmount = money //退款金额,个位为分
	ph.RefundReason = c.Param("operator") //退款原因

	ip := c.ClientIP()
	fmt.Println("ip="+ip)

	// 拉卡拉退款
	rtn, err := pay.UnifiedRefund(&ph, ip)
	if err != nil {
		zap.L().Error(err.Error())
		base.ResponseErrorMsg(c, err.Error())
		return
	}

	base.ResponseSuccess(c, rtn)
}

// OrderState 拉卡拉支付查询
// @Summary 拉卡拉支付查询
// @Description 拉卡拉支付查询
// @Tags 拉卡拉支付查询
// @Accept application/json
// @Produce application/json
// @Param   orderId   path   string   true	  "订单号"
// @Param language header string ture "语言类型 zh-CN简体中文 en-US英文 ja-JP日文 默认中文"
// @Success 200
// @router /api/v1/pay/order_state/{orderId} [get]
func (l *PayController) OrderState(c *gin.Context) {

	orderID := c.Param("orderId")

	// 拉卡拉支付查询
	rtn, err := pay.OrderState(orderID)
	if err != nil {
		zap.L().Error(err.Error())
		base.ResponseErrorMsg(c, err.Error())
		return
	}

	base.ResponseSuccess(c, rtn)
}